import { Request, Response } from 'express';
import { contributorsService } from './service';

function hasFeature(vendor: any, featureKey: string): boolean {
  return vendor?.features?.includes(featureKey);
}

export const contributorsController = {
  // Contributors
  async addContributor(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'contributor_roles')) {
      return res.status(403).json({ error: 'Contributor management not available on your plan.' });
    }
    const { data, error } = await contributorsService.addContributor(vendor.id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.status(201).json({ success: true, data });
  },

  async listContributors(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'contributor_roles')) {
      return res.status(403).json({ error: 'Contributor management not available on your plan.' });
    }
    const { data, error } = await contributorsService.listContributors(vendor.id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  async updateContributor(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'contributor_roles')) {
      return res.status(403).json({ error: 'Contributor management not available on your plan.' });
    }
    const { id } = req.params;
    const { data, error } = await contributorsService.updateContributor(vendor.id, id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  async deleteContributor(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'contributor_roles')) {
      return res.status(403).json({ error: 'Contributor management not available on your plan.' });
    }
    const { id } = req.params;
    const { data, error } = await contributorsService.deleteContributor(vendor.id, id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // Discounts
  async createDiscount(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'discount_rules')) {
      return res.status(403).json({ error: 'Discount management not available on your plan.' });
    }
    const { data, error } = await contributorsService.createDiscount(vendor.id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.status(201).json({ success: true, data });
  },

  async listDiscounts(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'discount_rules')) {
      return res.status(403).json({ error: 'Discount management not available on your plan.' });
    }
    const { type, active, applies_to, target_id } = req.query;
    const filters: any = { type, active, applies_to, target_id };
    const { data, error } = await contributorsService.listDiscounts(vendor.id, filters);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  async applyDiscounts(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'discount_rules')) {
      return res.status(403).json({ error: 'Discount management not available on your plan.' });
    }
    const { customer_id, product_ids } = req.query;
    const productIds = Array.isArray(product_ids) ? product_ids : (product_ids ? [product_ids] : []);
    const { data, error } = await contributorsService.applyDiscounts(vendor.id, customer_id, productIds);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  async updateDiscount(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'discount_rules')) {
      return res.status(403).json({ error: 'Discount management not available on your plan.' });
    }
    const { id } = req.params;
    const { data, error } = await contributorsService.updateDiscount(vendor.id, id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  async deleteDiscount(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'discount_rules')) {
      return res.status(403).json({ error: 'Discount management not available on your plan.' });
    }
    const { id } = req.params;
    const { data, error } = await contributorsService.deleteDiscount(vendor.id, id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  }
}; 